﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;

using MicroRWD.Common;
using MicroRWD.MFIC;

namespace MicroRWD.UI.MFIC
{
    public partial class Mifare_MemoryPanel : UserControl
    {
        #region Private Classes

        private class ReadWorker
        {
            // TODO: Move C_LAST_BLOCK to a common constants location
            private const byte C_LAST_BLOCK = 0x3F;
            private byte startAddress = 0x00;
            private byte endAddress = C_LAST_BLOCK;
            private bool keyType = false;
            private byte keyCodeNumber = 0x00;

            private Mifare_MemoryPanel parent;
            private ReadCardProgress progress;

            public ReadWorker(Mifare_MemoryPanel _parent, ReadCardProgress _progress, byte _startAddress, byte _endAddress, bool _keyType, byte _keyCodeNumber)
            {
                parent = _parent;
                progress = _progress;
                startAddress = ((_startAddress >= 0x00) && (_startAddress <= _endAddress) && (_startAddress <= (byte)C_LAST_BLOCK)) ? _startAddress : (byte)0x00;
                endAddress = ((_endAddress >= 0x00) && (_endAddress >= _startAddress) && (_endAddress <= C_LAST_BLOCK)) ? _endAddress : C_LAST_BLOCK;
                keyType = _keyType;
                keyCodeNumber = _keyCodeNumber;
            }

            public void DoWork()
            {
                try
                {
                    // Clear the listview
                    for (byte block = 0x00; block <= C_LAST_BLOCK; ++block)
                    {
                        // Clear data
                        parent.UpdateBlock(block, new string[] { "", "", "", "" }, new string[] { "", "", "", "" });
                    }

                    // Read card block by block
                    for (byte block = startAddress; (block <= endAddress) && !progress.Cancel; ++block)
                    {
                        // Delegate read to reader
                        byte[] reply = Program.Controller.Reader.CmdCardBlockRead(block, keyType, keyCodeNumber);

                        // Check for reply
                        if (reply.Length > 0)
                        {
                            // Check for data
                            if (reply.Length == 17)
                            {
                                
                                string[] hex = new string[4] {"", "", "", ""};
                                string[] ascii = new string[4] { "", "", "", "" };
                                byte[] data = new byte[4];

                                // Extract data
                                for (int i = 0; i < 4; i++)
                                {
                                    Array.Copy(reply, i*4 + 1, data, 0, data.Length);
                                    hex[i] = String.Join(string.Empty, Array.ConvertAll(data, b => b.ToString("X2") + ' '));
                                    ascii[i] = String.Join(string.Empty, Array.ConvertAll(data, b => ((b > 31) && (b < 127)) ? ((char)b).ToString() + ' ' : ". "));
                                }

                                // Copy data into list view
                                parent.UpdateBlock(block, hex, ascii);
                            }
                            else
                            {
                                // Copy data into list view
                                parent.UpdateBlock(block, new string[] { "<NO DATA>", "<NO DATA>", "<NO DATA>", "<NO DATA>" }, new string[] { "", "", "", "" });
                            }
                        }
                        else
                        {
                            // Clear data
                            parent.UpdateBlock(block, new string[] { "", "", "", "" }, new string[] { "", "", "", "" });
                        }

                        // Update progress bar
                        if (endAddress == 0)
                        {
                            parent.UpdateProgress(block * 100 / 1); // prevent divide by zero
                        }
                        else
                        {
                            parent.UpdateProgress(block * 100 / endAddress);
                        }
                    }

                    // Close progress dialog
                    parent.WorkerDone();
                }
                catch (Exception ex)
                {
                    // Error
                    Log.Error(ex.ToString());
                }
            }
        }
        
        #endregion

        #region Private Properties

        // Create progress dialog
        private ReadCardProgress progress;
        private ReadMifareCardMemory readCardMemory;
        private WriteMifareCardMemory writeCardMemory;

        #endregion

        public Mifare_MemoryPanel()
        {
            InitializeComponent();

            // TODO: Load Card Memory Definitions
            cardListView.Items.Add(new ListViewItem(new string[] { "00", "-- -- -- -- ", ". . . . ", "Serial no. +" }));
            cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "Mfr. data" }));
            cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
            cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));

            for (byte i = 0x01; i <= 0x3F; ++i)
            {
                if ((i + 1) % 4 == 0)
                {
                    cardListView.Items.Add(new ListViewItem(new string[] { i.ToString("X2"), "-- -- -- -- ", ". . . . ", "Keys A/B +" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "Access Bits" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
                }
                else
                {
                    cardListView.Items.Add(new ListViewItem(new string[] { i.ToString("X2"), "-- -- -- -- ", ". . . . ", "User Data" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
                    cardListView.Items.Add(new ListViewItem(new string[] { "", "-- -- -- -- ", ". . . . ", "" }));
                }
            }

            // Select first item
            cardListView.Items[0].Selected = true;
        }

        #region Public Methods

        public void readCard(byte _startAddress, byte _endAddress, bool _keyType, byte _keyCodeNumber)
        {
            // Create progress dialog
            progress = new ReadCardProgress();

            // Create worker to do work asynchronously
            ReadWorker worker = new ReadWorker(this, progress, _startAddress, _endAddress, _keyType, _keyCodeNumber);
            Thread t = new Thread(worker.DoWork);
            t.Start();
            progress.ShowDialog();
            t.Join();
        }

        public byte[] readCard(byte _blockAddress, bool _keyType, byte _keyCodeNumber)
        {
            return Program.Controller.Reader.CmdCardBlockRead(_blockAddress, _keyType, _keyCodeNumber);
        }

        public void writeCard(byte _block, bool _keyType, byte _keyCodeNumber, byte[] _data)
        {
            // If data is valid
            if ((_data != null) && (_data.Length == 16))
            {
                // Write to card
                byte[] reply = Program.Controller.Reader.CmdCardBlockWrite(_block, _keyType, _keyCodeNumber, _data);
                if (reply.Length > 0)
                {
                    if (reply.Length == 1)
                    {
                        byte ack = reply[0];

                        if ((ack & 0xFE) == 0x86)
                        {
                            string[] hex = new string[4] { "", "", "", "" };
                            string[] ascii = new string[4] { "", "", "", "" };
                            byte[] dataPart = new byte[4];

                            // Extract data
                            for (int i = 0; i < 4; i++)
                            {
                                Array.Copy(_data, i * 4, dataPart, 0, dataPart.Length);
                                hex[i] = String.Join(string.Empty, Array.ConvertAll(dataPart, b => b.ToString("X2") + ' '));
                                ascii[i] = String.Join(string.Empty, Array.ConvertAll(dataPart, b => ((b > 31) && (b < 127)) ? ((char)b).ToString() + ' ' : ". "));
                            }

                            // Copy data into list view
                            UpdateBlock(_block, hex, ascii);
                        }
                    }
                }
            }
        }

        public string getDescription(byte _block)
        {
            string description = "";
            if (cardListView != null)
            {
                int row = _block * 4;
                int subitem = 3;
                description = cardListView.Items[row++].SubItems[subitem].Text;
                description += cardListView.Items[row++].SubItems[subitem].Text;
                description += cardListView.Items[row++].SubItems[subitem].Text;
                description += cardListView.Items[row++].SubItems[subitem].Text;
            }
            else
            {
                description = "";
            }

            return description;
        }

        #endregion

        #region Private Methods

        private void UpdateProgress(int _value)
        {
            this.UIThread(() => progress.Value = _value);
        }
        
        private void UpdateBlock(int _block, string[] _hex, string[] _ascii)
        {
            if ((_hex.Length == 4) && (_ascii.Length == 4))
            {
                int row;
                row = _block * 4;
                this.UIThread(() => cardListView.Items[row + 0].SubItems[1].Text = _hex[0]);
                this.UIThread(() => cardListView.Items[row + 0].SubItems[2].Text = _ascii[0]);

                this.UIThread(() => cardListView.Items[row + 1].SubItems[1].Text = _hex[1]);
                this.UIThread(() => cardListView.Items[row + 1].SubItems[2].Text = _ascii[1]);

                this.UIThread(() => cardListView.Items[row + 2].SubItems[1].Text = _hex[2]);
                this.UIThread(() => cardListView.Items[row + 2].SubItems[2].Text = _ascii[2]);

                this.UIThread(() => cardListView.Items[row + 3].SubItems[1].Text = _hex[3]);
                this.UIThread(() => cardListView.Items[row + 3].SubItems[2].Text = _ascii[3]);
            }
            else
            {
                int row;
                for (int i = 0; i < 4; ++i)
                {
                    row = _block * 4 + i;
                    this.UIThread(() => cardListView.Items[row].SubItems[1].Text = "");
                    this.UIThread(() => cardListView.Items[row].SubItems[2].Text = "");
                }
            }
        }

        private void WorkerDone()
        {
            this.UIThread(() => progress.Close());
        }

        private void writeToCardDialog()
        {
            int selectedIndex = 0;
            if (cardListView.IsHandleCreated)
            {
                if (cardListView.SelectedIndices.Count > 0)
                {
                    selectedIndex = cardListView.SelectedIndices[0];
                }
                else
                {
                    selectedIndex = 0;
                }
            }
            else
            {
                selectedIndex = 0;
            }
            byte block = (byte)(selectedIndex/4);

            writeCardMemory = new WriteMifareCardMemory(this, block);
            writeCardMemory.ShowDialog();
        }

        #region Event Handler

        private void Mifare_MemoryPanel_Load(object sender, EventArgs e)
        {
            // Register for status update events unless in design mode
            if (!DesignMode && (Program.Controller != null))
            {
                Program.Controller.Reader.StatusUpdateEvent += MyStatusChangedEventHandler;
            }
        }

        // Handler for status changed events
        private void MyStatusChangedEventHandler(object sender, StatusUpdateEventArgs args)
        {
            // Update Serial Number text with the latest value
            this.UIThread(() => serialNumberValueLabel.Text = Program.Controller.Reader.LastMifareUidStr);
        }

        private void readCardButton_Click(object sender, EventArgs e)
        {
            // Create read card dialog
            readCardMemory = new ReadMifareCardMemory(this);
            readCardMemory.ShowDialog();
        }

        private void writeToCardButton_Click(object sender, EventArgs e)
        {
            writeToCardDialog();
        }

        private void cardListView_DoubleClick(object sender, EventArgs e)
        {
            writeToCardDialog();
        }
                
        #endregion

        #endregion
    }
}
